#!/usr/bin/env python3
# JS8Spotter Data Siphon. Visit https://kf7mix.com/js8spotter.html for information
# For gathering data from multiple running instances of JS8Spotter, into a central database file
#
# MIT License, Copyright 2025 Joseph D Lyman KF7MIX -- Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation files
# (the "Software"), to deal in the Software without restriction, including without limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies
# of the Software, and to permit persons to whom the Software is furnished to do so, subject to the following conditions: The above copyright notice and this permission notice shall be
# included in all copies or substantial portions of the Software. The Software IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
# WARRANTIES OF MERCHANTABILITY, FITNESS OR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

import sqlite3
import time
import os


### Variables

# an identifier for this node, in case you want to sort or organize siphoned data
# an integer greater than zero, which should be unique to this instance of JS8Spotter
node_id = 100

# how often to scan for new data, seconds. 10 is default, lower amounts are not helpful as JS8Call has 15sec normal speed frames
scan_delay = 10

# full or relative path to a single running instance database
# you will run multiple instances of this siphon program, one for each running spotter instance
spotter_instance_dbfile = 'js8spotter.db'

# this should be a _full_ path to a single aggregate database, which will be shared among all instances of siphon
# use the included tool to create a blank file, or use the included blank
aggregate_siphon_dbfile = 'siphon.db'


### Establish database connections

conn = sqlite3.connect(spotter_instance_dbfile)
c_source = conn.cursor()

conn1 = sqlite3.connect(aggregate_siphon_dbfile)
c_dest = conn1.cursor()


### Functions

def perform_siphon():
    get_latest_sql = "SELECT * FROM activity WHERE (spotdate > DATETIME('now','-"+str(scan_delay)+" Second'))"
    c_source.execute(get_latest_sql)
    activity_records = c_source.fetchall()

    if not activity_records:
        print(".", end='', flush=True)
    else:
        print(f"New data found: {time.ctime()}")
        for record in activity_records:
            add_records_sql = "INSERT INTO activity(node_id,type,value,dial,snr,call,spotdate,freq,offset,speed) VALUES (?,?,?,?,?,?,?,?,?,?)"
            c_dest.execute(add_records_sql, [node_id,record[2],record[3],record[4],record[5],record[6],record[7],record[8],record[9],record[10]])
            conn1.commit()

def activate_siphon(interval_seconds):
    next_execution_time = time.time() + interval_seconds

    while True:
        current_time = time.time()
        if current_time >= next_execution_time:
            perform_siphon()
            next_execution_time = current_time + interval_seconds

        sleep_duration = max(0, next_execution_time - time.time())
        time.sleep(sleep_duration)

### Begin

if __name__ == "__main__":
    print(f"Scanning for new JS8Spotter data every {scan_delay} seconds. CTRL+C to quit.")
    activate_siphon(scan_delay)

